%% Lomb-Scargle Power Spectrum
% Martin H. Trauth 10 June 2021
%
% We first clear the workspace and the command window. We then close all
% figure windows.
clear, clc, close all

%%
% We then define the noise level, i.e. the standard deviation of the
% Gaussian noise, and create an unevenly spaced time vector t.
Ns = 5;
rng(0)
t = 100*rand(1000,1);
t = sortrows(t);

%%
% We then create a composite signal of multiple sine waves with different
% periods T, different amplitudes A and additive Gaussian noise Ns.
rng(0)
x =   2*sin(2*pi*t/50) + ...
        sin(2*pi*t/15) + ...
    0.5*sin(2*pi*t/5) + ...
     Ns*randn(size(t));

figure('Position',[100 700 800 400],...
    'Color',[1 1 1])
axes('XGrid','On',...
    'YGrid','On'), hold on
line(t,x,...
    'LineWidth',1)

%%
% We can calculate the normalized Lomb-Scargle power spectrum and false
% alarm probabilities using a MATLAB version of the original FORTRAN code
% of Scargle (1981).
int = mean(diff(t));
ofac = 4; hifac = 1;
f = ((2*int)^(-1))/(length(x)*ofac): ...
    ((2*int)^(-1))/(length(x)*ofac): ...
    hifac*(2*int)^(-1);
f = f';

for k = 1:length(f)
    wrun = 2*pi*f(k);
    tau = (1/2*wrun)*atan2(sum(sin(2*wrun*t)), ...
       sum(cos(2*wrun*t)));
    px(k) = 1/(2*var(x)) * ...
       (...
       (sum((x - mean(x)).*cos(wrun*(t-tau)))).^2 ...
       /sum((cos(wrun*(t-tau))).^2) + ...
       (sum((x - mean(x)).*sin(wrun*(t-tau)))).^2 ...
       /sum((sin(wrun*(t-tau))).^2) ...
       );
end
px = px';

prob = 1-(1-exp(-px)).^(2*length(x));

figure('Position',[100 600 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 0.5],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(f,px,...
    'LineWidth',1)
xlabel('Frequency')
ylabel('Power')

figure('Position',[100 500 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 0.5],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(f,prob,...
    'LineWidth',1)
xlabel('Frequency')
ylabel('False Alarm Probability')

%%
% We can calculate levels that correspond to false-alarm probabilities
% confid.
m = floor(0.5*ofac*hifac*length(x));
effm = 2*m/ofac;
confid = [0.5 0.1 0.01];
levels = log((1-confid.^(1/effm)).^(-1));

figure('Position',[100 400 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 0.5],...
    'XGrid','On',...
    'YGrid','On')
line(f,px,...
    'LineWidth',1)
line(f,levels.*ones(size(f)),...
    'LineStyle','--')
text(0.3*ones(size(confid)),levels'-.5, ...
    [repmat('P_{fa} = ',size(confid')) num2str(confid')])
xlabel('Frequency')
ylabel('Power')

%%
% We can calculate the normalized Lomb-Scargle power spectrum and false
% alarm probabilities using the MATLAB function plomb. Annotate the levels
% that correspond to false-alarm probabilities confid.
[px,f,levels] = plomb(x,t,'normalized',...
    'Pd',confid);

figure('Position',[100 300 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 0.5],...
    'XGrid','On',...
    'YGrid','On')
line(f,px,...
    'LineWidth',1)
line(f,levels'.*ones(size(f)),...
    'LineStyle','--')
text(0.3*ones(size(confid)),levels'-.5, ...
    [repmat('P_{fa} = ',size(confid')) num2str(confid')])
xlabel('Frequency')
ylabel('Power')

